/* TERMINAL.C: A simple terminal emulation */
/* By: Martin Grotegut 1989-1990           */
/* Demonstrates the use of SERISR.C        */

/* Include's */
#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <conio.h>

/* Define's */
/* 8250 register defines (add these offsets to base addr)		*/
#define RBR 0		   	/* Receive Buffer Register READ ONLY	*/
#define THR 0		   	/* Transmitter Hold Register WRITE ONLY */
#define IER 1		   	/* Interrupt Enable Register		*/
#define IIR 2		   	/* Interrupt ID Register		*/
#define LCR 3		   	/* Line Control Register		*/
#define MCR 4		   	/* Modem Control Register		*/
#define LSR 5		   	/* Line Status Register			*/
#define MSR 6		   	/* Modem Status Register		*/
#define DLAR 0		   	/* Divisor Low (Bits 0-7)		*/
#define DHAR 1		   	/* Divisor High (Bits 8-15)		*/

#define TRUE 1
#define FALSE 0
#define REG register
#define UCHAR unsigned char
#define UWORD unsigned

/* Function prototype's */
void init_rs232(void);
void reset_rs232(void);
void init_base_addr(void);
void (interrupt far *old_intC)(void);
UWORD (interrupt far *old_int14)(void);
void interrupt far new_intC(unsigned, unsigned, unsigned, unsigned, unsigned,
			    unsigned, unsigned, unsigned, unsigned, unsigned,
			    unsigned, unsigned, unsigned);
UWORD interrupt far new_int14(unsigned, unsigned, unsigned, unsigned, unsigned,
			      unsigned, unsigned, unsigned, unsigned, unsigned,
			      unsigned, unsigned, unsigned);

/* Global storage */
static UWORD baud_rate_divisor[8] = {1047, 768, 384, 192, 96, 48, 24, 12};
UWORD	rs232_base[4];			/* 8250 Base Addresses array */
UCHAR	com_buff_head,
	com_buff_tail;
int	com_buff[256];			/* 256 chars + status bytes */


main()
{
char	kb_char,			/* Char from keyboard */
	com_char,			/* "    "    COM1:    */
	ch;
int	kb_stat,			/* Status of keyboard */
	com_stat;		        /* "      "  COM1:    */

	init_rs232();			/* Enable interrupt */

	_bios_serialcom(_COM_INIT, 0, _COM_1200|_COM_CHR7|_COM_STOP2|_COM_NOPARITY);

	printf("\nNow in terminal mode. Press ESC to exit.\n");
	do
	{
	 kb_stat = _bios_keybrd( _KEYBRD_READY );
	 if( kb_stat )
	 {
	  kb_char = (char) _bios_keybrd( _KEYBRD_READ );
	  if( kb_char )
	  {
	    com_stat = _bios_serialcom( _COM_SEND, 0, kb_char );
	    putchar( kb_char );
	    if( kb_char == '\r' )        /* <RETURN>? */
	    {
	     com_stat = _bios_serialcom( _COM_SEND, 0, '\n' );  /* Send <NL> */
	     putchar( '\n' );
	    }
	  }
	 }

	 if( com_stat = (_bios_serialcom(_COM_STATUS, 0, 0) & 0x0100) ) /* Character(s) waiting ? */
	 {
	  com_char = _bios_serialcom( _COM_RECEIVE, 0, 0 );
	  if( com_char && com_char != 4 ) /* Remove any NUL's and EOT's */
	   putchar( com_char );
	 }
	}
	while( kb_char != 27 );		/* ESC halts program run */

	reset_rs232();
	printf("\nExit to DOS\n");
}


void interrupt far new_intC(unsigned es, unsigned ds, unsigned di,
			    unsigned si, unsigned bp, unsigned sp,
			    unsigned bx, unsigned dx, unsigned cx,
			    unsigned ax, unsigned ip, unsigned cs,
			    unsigned flags)
{
	_disable();
	if((inp(0x3F8+IIR) & 0x06) == 4) /* Test for Data Read Intr */
	{
	 /* Read char from RCV buffer reg. and setup status in AH */
	 com_buff[com_buff_head++]=(inp(0x3F8+RBR)&0xFF) | ((inp(0x3F8+LSR)&0x1E)<<8);
	 outp(0x20, 0x20);		/* Tell 8259 End-of-intr */
	}
	_enable();
}


unsigned interrupt far new_int14(unsigned es, unsigned ds, unsigned di,
				 unsigned si, unsigned bp, unsigned sp,
				 unsigned bx, unsigned dx, unsigned cx,
				 unsigned ax, unsigned ip, unsigned cs,
				 unsigned flags)
{
UWORD	tmp_base;

	_disable();
	dx &= 0x03;			/* 4 COM ports allowed */
	if( tmp_base = rs232_base[dx] ) /* If 8250 exists */
	{
	 switch(ax >> 8)
	 {
	  case 0x00:			/* Set comm. params */
	 	 outp(tmp_base+LCR, 0x80); /* Set DLAB bit */
		 outp(tmp_base+DLAR, baud_rate_divisor[ax>>5]&0xFF);
	 	 outp(tmp_base+DHAR, baud_rate_divisor[ax>>5]>>8);
		 outp(tmp_base+LCR, ax & 0x1F);	/* Set remaining bits */
	  case 0x03:			/* Status request */
	  	 /* Get status */
		 ax = (inp(tmp_base+LSR)<<8) | (inp(tmp_base+MSR)&0xFF);
		 if( com_buff_head == com_buff_tail)	/* Chars in buffer ? */
		  ax &= 0xFEFF;		/* Say buffer empty */
		 else
		  ax |= 0x0100;		/* Fake data ready */
		 break;

	  case 0x01:			/* Send char in AL */
		 ax &= 0xFF;
		 while( ( inp( tmp_base+LSR ) & 0x20 ) == 0 )
		  ;			/* Wait for THRE empty */
		 outp(tmp_base+THR, ax); /* Send char */
		 ax |= inp(tmp_base+LSR)<<8; /* 'Or' in status */
		 break;

	  case 0x02:			/* Receive char */
		 if( com_buff_head != com_buff_tail) /* Buffer not empty */
		  ax = com_buff[com_buff_tail++]; /* Get char & status */
		 else
		  ax = (inp(tmp_base+LSR)<<8) | 0x8000; /* Time out */
		 break;

	 }
	}
	_enable();
	return( ax );
}


/* Enable the 8250 UART to generate interrupts */
void init_rs232(void)
{
	_disable();
	com_buff_head = com_buff_tail = 0;
	init_base_addr();		/* Get 8250 base addresses */

	/* Get old routine vectors */
	old_intC  = _dos_getvect( 0x0C ); /* Hex C; COM1: */
	old_int14 = _dos_getvect( 0x14 );

	/* Point to new routines */
	_dos_setvect(0x0C, new_intC);
	_dos_setvect(0x14, new_int14);

	outp(0x3F8+LCR,inp(0x3F8+LCR)&0x7F); /* Reset Div. Latch Access Bit */
	outp(0x3F8+IER,inp(0x3F8+IER)|0x01); /* Enable Rec. data int. 	    */
	outp(0x3F8+MCR,inp(0x3F8+MCR)|0x0B); /* Set DTR, RTS, Intr enable   */
	outp(0x21, inp(0x21) & 0xEF);	/* Enable 8259 COM1 interrupts      */
	inp(0x3F8);			/* RCV twice to clear old char and  */
	inp(0x3F8);			/* interrupt that may be pending    */
	_enable();
}


/* Fills base addresses array with proper 8250 base adresses */
/* If a base address is zero, an 8250 was not found          */
void init_base_addr(void)
{
UWORD REG far *ptr;
UWORD REG i;

	ptr = (UWORD far *) 0x00000400; /* DOS copy of 8250 base addr */
	for(i = 0; i < 4; i++)
	 rs232_base[i] = *ptr++;
}


void reset_rs232(void)
{
	_disable();
	outp(0x3F8+IER, 0);		/* Disable all 8250 interrupts	*/
	outp(0x3F8+MCR, 0);		/* Reset DTR, RTS, Intr enable	*/
	outp(0x21, inp(0x21) | 0x10);	/* Disable 8259 COM1 interrupts */
	/* Restore interrupt vectors */
	_dos_setvect(0x0C, old_intC);
	_dos_setvect(0x14, old_int14);
	inp(0x3F8);			/* RCV twice to clear old char and  */
	inp(0x3F8);			/* interrupt that may be pending    */
	_enable();
}

